#!/bin/bash

# acuppd.sh
#
# Usage:
#    acuppd.sh [ -l LOCK_ID | --lock LOCK_ID ]
#    acuppd.sh -c LOCK_ID | --clean LOCK_ID
#    acuppd.sh -h | --help
#
# Description:
#    This script gathers relevant PA files, logs
#    its progress, and tars everything up, using
#    gzip for compression (-z option of tar).  It
#    may be run in two different ways, locked (run
#    from a script) and not locked (run manually).
#
#    LOCKED:  It is lockable for a given LOCK_ID.
#    In other words, if it is called again with the
#    same LOCK_ID before cleanup has been requested
#    with said LOCK_ID, it will simply exit with an
#    error RC.  Naming conventions of files will keep
#    themselves seperate from other locks by using
#    the lock id in them.  For example, the archive
#    file will be named acuppd.$LOCK_ID.tgz.
#
#    UNLOCKED:  If it is run without the --lock option,
#    as it is assumed will be done when manually run, no
#    locking is performed, --clean can not be called,
#    the temporary directories will be removed, but
#    it is the requirement of the caller to then
#    remove the archive file.  If it is manually run
#    again before the previous .tgz file is deleted,
#    the script will remove it before creating the
#    new one.  The archive file will be named
#    acuppd.tgz.
#
# Command Line Parameters:
#    --lock LOCK_ID: locks the script from being run again with LOCK_ID before --clean is called
#    --clean LOCK_ID: removes tar file, any owned temporary directories, and the PID lock for LOCK_ID
#    --help: displays usage information and exits
#
# Return Codes
#    0: normal script termination; no unrecoverable errors
#    1: invalid command line parameters
#    2: could not locate hmcfunctions script file
#    3: script is currently locked by another process for given lock id
#    4: temporary archive directory already exists
#    5: insufficient permissions to create lock file (locked state only)
#
# Module History
#    00  10/15/02   J. Jenks      - Initial Release
#    01  10/16/02   J. Jenks      - Usage changes: --name=NAME is now --name NAME
#    02  11/24/03   J. Jenks      - Updated ". hmcfunctions" call to work on both
#                                   development and production environments
#    03  01/12/04   J. Jenks      - Uses queryFileLocation to find console data files
#    04  03/25/04   J. Jenks      - Added actwcud.dat to the archiving list
#    05  03/26/04   L. Brocious   - Add Unified JVM support; capture "top -bn1" output;
#                                   capture javacore files (from IBM JVM)
#    06  03/30/04   J. Jenks      - Added /tmp/hmc/ud to the archiving list
#    07  04/12/04   J. Jenks      - Removed --name parameter option; archive is always acuppd.tgz
#    08  04/12/04   J. Jenks      - Added --clean option to cleanup archiving files
#    09  04/12/04   J. Jenks      - Implemented locking mechanism so the script can not be
#                                   run multiple times concurrently.  Running the script while
#                                   locked will result in exit with RC=3.  Running the script
#                                   with the --clean parameter will remove the lock.
#    10  05/24/04   K. Schroeder  - Add collection of files in data/core directory.
#    11  11/16/04   P. Callaghan  - Add running of "showTraceBuff all" script.
#    12  11/29/04   J. Jenks      - Modified script to be lockable on a per caller basis,
#                                   or not locked at all for running manually.
#    13  11/29/04   J. Jenks      - Add collection of CONSOLE_PATH/data/rcs/rcsControl.log
#


# -----------
# subroutines
# -----------

# subroutine for outputting help
echo_help() {
   if [ -n "$ERROR" ]; then
      echo "Error: $ERROR" >&2
      echo >&2
   fi
   echo "Usage: archives PA files into gzipped tar (tar -cz)"
   echo "   `basename $0` [ -l LOCK_ID | --lock LOCK_ID ]"
   echo "   `basename $0` -c LOCK_ID | --clean LOCK_ID"
   echo "   `basename $0` -h | --help"
   echo
   echo "Parameters:"
   echo "   --lock LOCK_ID: locks the script from being run again before --clean is called on LOCK_ID"
   echo "   --clean LOCK_ID: removes tar file, any owned temporary directories, and the PID lock file for LOCK_ID"
   echo "   --help: displays this help"
   echo
}

# subroutine for checking if there were errors while collecting data
check_err() {
   if [ -s err.tmp ]; then
      echo "errors follow." >>$LOG
      cat -n err.tmp >>$LOG
   else
      echo "done." >>$LOG
   fi
   rm -f err.tmp
   echo >>$LOG
}


# -----------------------------------------
# perform parameter collection and checking
# -----------------------------------------

# check for parameters and output help if required
until [ -z "$1" -o -n "$HELP" ]; do
   case "$1" in
      -l | --lock     )
         if [ -n "$CLEAN" ]; then
            HELP="on"
            ERROR="Only --clean OR --lock may be specified!"
         else
            LOCK="on"
            shift
            LOCKID="$1"
            if [ -z "$LOCKID" ]; then
               HELP="on"
               ERROR="A lock id must be specified with the --lock option!"
            fi
         fi
      ;;
      -c | --clean    )
         if [ -n "$LOCK" ]; then
            HELP="on"
            ERROR="Only --clean OR --lock may be specified!"
         else
            CLEAN="on"
            shift
            LOCKID="$1"
            if [ -z "LOCKID" ]; then
               HELP="on"
               ERROR="A lock id must be specified with the --clean option!"
            fi
         fi
      ;;
      -h | --help     )
         HELP="on"
      ;;
      *               )
         HELP="on"
         ERROR="Invalid parameter ( $1 )!"
      ;;
   esac
   shift
done

if [ -n "$HELP" ]; then
   echo_help
   if [ -n "$ERROR" ]; then
      exit 1
   else
      exit 0
   fi
fi


# ------------------------------------
# find necessary archiving directories
# ------------------------------------

# attempt to locate hmcfunctions script
if [ -z "$CONSOLE_PATH" ]; then
   CONSOLE_PATH="$HWMCAHOME/"
fi
FOUND=
for FUNCDIR in "" "$CONSOLE_PATH/" "$CONSOLE_PATH/native/scripts/"; do
   . ${FUNCDIR}hmcfunctions >/dev/null 2>&1 && FOUND=true && break 1
done
if [ -z "$FOUND" ]; then
   echo "Error: Could not locate hmcfunctions script file!" >&2
   exit 2
fi

# find locations and names of archiving files
if [ -n "$LOCKID" ]; then
   TEMP_DIR="`queryFileLocation acuppd`acuppd.${LOCKID}"
   ARCHIVE_FILE="`queryFileLocation acuppd.tgz`acuppd.${LOCKID}.tgz"
   PID_FILE="`queryFileLocation acuppd.pid`acuppd.${LOCKID}.pid"
else
   TEMP_DIR="`queryFileLocation acuppd`acuppd"
   ARCHIVE_FILE="`queryFileLocation acuppd.tgz`acuppd.tgz"
   PID_FILE="`queryFileLocation acuppd.pid`acuppd.pid"
fi


# ----------------------------------
# manage script cleaning and locking
# ----------------------------------

# if this is a call to clean up old files, do so and exit
if [ -n "$CLEAN" ]; then
   #-10 start
   rm -f $CONSOLE_PATH/data/core/* >/dev/null 2>&1
   #-10 end
   rm -f $ARCHIVE_FILE >/dev/null 2>&1
   rm -f $PID_FILE >/dev/null 2>&1
   exit 0
fi

# we are here, so the request is to archive... check for lock if necessary
if [ -n "$LOCK" ]; then
   if [ -e $PID_FILE ]; then
      # script is locked, so output error and exit
      echo "Exiting because script is already running on pid `cat $PID_FILE`!" >&2
      exit 3
   else
      # script is not locked, so attempt to lock it
      if ! echo -n "$$" 2>/dev/null >$PID_FILE; then
         echo "Could not write lock file!  Insufficient permissions (`dirname $PID_FILE`)?" >&2
         exit 5
      fi
   fi
fi


# -------------
# setup archive
# -------------

# create temp dir for storing PA files
if [ -e $TEMP_DIR ]; then
   echo "Exiting because temporary archive directory already exists!  Is another manual instance running?" >&2
   exit 4
fi
mkdir $TEMP_DIR
pushd $TEMP_DIR >/dev/null
mkdir -p pa.commands pa.logs pa.dumps pa.data pa.tmp/hmc pa.etc/sysconfig

# setup log file
LOG="$TEMP_DIR/acuppd.log"
echo "acuppd log file" >$LOG
echo "---------------" >>$LOG
echo >>$LOG
echo "Started:" >>$LOG
date >>$LOG
echo >>$LOG
echo "Notes about this archive's contents:" >>$LOG
echo "   The output of commands, in general, are put in files named as the command followed by [.<options>].out.  .<options> is the list of options used with the command.  If none are used, it is left out.  For example, the output of the command ps will be put in a file called ps.out, whereas ps -A will be ps.A.out.  This applies to all commands except for ls, whose output is named by ls[.<options>].<name of base directory>.  For example, the output of ls -R /proc is named ls.R.proc." >>$LOG
echo >>$LOG


# ----------------------------
# generate files from commands
# ----------------------------

# ps -A
echo -n "Running ps -A ... " >>$LOG
ps -A >./pa.commands/ps.A.out 2>err.tmp
check_err

# ls -R /proc
echo -n "Running ls -R /proc ... " >>$LOG
ls -R /proc >./pa.commands/ls.R.proc 2>err.tmp
check_err

#-05 begin
# top -bn1
echo -n "Running top -bn1 ... " >>$LOG
top -bn1 >./pa.commands/top.bn1.out 2>err.tmp
check_err
#-05 end

# ls -R /dev
echo -n "Running ls -R /dev ... " >>$LOG
ls -R /dev >./pa.commands/ls.R.dev 2>err.tmp
check_err

# ipcs
echo -n "Running ipcs ... " >>$LOG
ipcs >./pa.commands/ipcs.out 2>err.tmp
check_err

# iqzzqtcs
echo -n "Running iqzzqtcs ... " >>$LOG
iqzzqtcs >./pa.commands/iqzzqtcs.out 2>err.tmp
check_err

# ifconfig
echo -n "Running ifconfig ... " >>$LOG
/sbin/ifconfig >./pa.commands/ifconfig.out 2>err.tmp
check_err

# netstat
echo -n "Running netstat ... " >>$LOG
netstat >./pa.commands/netstat.out 2>err.tmp
check_err

# netstat -r
echo -n "Running netstat -r ... " >>$LOG
netstat -r >./pa.commands/netstat.r.out 2>err.tmp
check_err

# netstat -l
echo -n "Running netstat -l ... " >>$LOG
netstat -l >./pa.commands/netstat.l.out 2>err.tmp
check_err

# showTraceBuf all
echo -n "Running showTraceBuf all ... " >>$LOG
showTraceBuf all >./pa.commands/showTraceBuf.all.out 2>err.tmp
check_err


# --------------------
# copy necessary files
# --------------------

# /var/log/messages*
echo -n "Copying /var/log/messages* ... " >>$LOG
cp /var/log/messages* ./pa.logs/ >err.tmp 2>&1
check_err

#-05 begin
if [ -n "`isUnified`" ]; then  # Only 1 console log file to capture
   # /var/log/hmc*
   echo -n "Copying /var/log/hmc* ... " >>$LOG
   cp /var/log/hmc* ./pa.logs/ >err.tmp 2>&1
   check_err
else                           # 2 console log files to capture
#-05 end
   # /var/log/manager*
   echo -n "Copying /var/log/manager* ... " >>$LOG
   cp /var/log/manager* ./pa.logs/ >err.tmp 2>&1
   check_err
   
   # /var/log/client*
   echo -n "Copying /var/log/client* ... " >>$LOG
   cp /var/log/client* ./pa.logs/ >err.tmp 2>&1
   check_err
fi #-05

#-13 begin
# CONSOLE_PATH/data/rcs/rcsControl.log
echo -n "Copying $CONSOLE_PATH/data/rcs/rcsControl.log ... " >>$LOG
cp $CONSOLE_PATH/data/rcs/rcsControl.log ./pa.logs/ >err.tmp 2>&1
check_err
#-13 end

# CONSOLE_PATH/core*
echo -n "Copying $CONSOLE_PATH/core* ... " >>$LOG
cp $CONSOLE_PATH/core* ./pa.dumps/ >err.tmp 2>&1
check_err

#-10 start
# CONSOLE_PATH/data/core/*
echo -n "Copying $CONSOLE_PATH/data/core/* ... " >>$LOG
cp $CONSOLE_PATH/data/core/* ./pa.dumps/ >err.tmp 2>&1
check_err
#-10 end

#-05 begin
# CONSOLE_PATH/javacore*
echo -n "Copying $CONSOLE_PATH/javacore* ... " >>$LOG
cp $CONSOLE_PATH/javacore* ./pa.dumps/ >err.tmp 2>&1
check_err
#-05 end

# CONSOLE_PATH/hs_err_pid*.log
echo -n "Copying $CONSOLE_PATH/hs_err_pid*.log ... " >>$LOG
cp $CONSOLE_PATH/hs_err_pid*.log ./pa.dumps/ >err.tmp 2>&1
check_err

# iqyvpd*
LOC="`queryFileLocation iqyvpd*`"
echo -n "Copying ${LOC}iqyvpd* ... " >>$LOG
cp ${LOC}iqyvpd* ./pa.data/ >err.tmp 2>&1
check_err

# actzuict.dat
LOC="`queryFileLocation actzuict.dat`"
echo -n "Copying ${LOC}actzuict.dat ... " >>$LOG
cp ${LOC}actzuict.dat ./pa.data/ >err.tmp 2>&1
check_err

# iqzzspr.dat
LOC="`queryFileLocation iqzzspr.dat`"
echo -n "Copying ${LOC}iqzzspr.dat ... " >>$LOG
cp ${LOC}iqzzspr.dat ./pa.data/ >err.tmp 2>&1
check_err

# persist/*
LOC="`queryFileLocation persist`"
echo -n "Copying ${LOC} ... " >>$LOG
cp -R ${LOC} ./pa.data/ >err.tmp 2>&1
check_err

# iqzdtrac.trm
LOC="`queryFileLocation iqzdtrac.trm`"
echo -n "Copying ${LOC}iqzdtrac.trm ... " >>$LOG
cp ${LOC}iqzdtrac.trm ./pa.data/ >err.tmp 2>&1
check_err

# actwcud.dat
LOC="`queryFileLocation actwcud.dat`"
echo -n "Copying ${LOC}actwcud.dat ... " >>$LOG
cp ${LOC}actwcud.dat ./pa.data/ >err.tmp 2>&1
check_err

# /tmp/hmc/ud
echo -n "Copying /tmp/hmc/ud ... " >>$LOG
cp -R /tmp/hmc/ud ./pa.tmp/hmc/ >err.tmp 2>&1
check_err

# /etc/host*
echo -n "Copying /etc/host* ... " >>$LOG
cp /etc/host* ./pa.etc/ >err.tmp 2>&1
check_err

# /etc/fstab
echo -n "Copying /etc/fstab ... " >>$LOG
cp /etc/fstab ./pa.etc/ >err.tmp 2>&1
check_err

# /etc/mtab
echo -n "Copying /etc/mtab ... " >>$LOG
cp /etc/mtab ./pa.etc/ >err.tmp 2>&1
check_err

# /etc/protocols
echo -n "Copying /etc/protocols ... " >>$LOG
cp /etc/protocols ./pa.etc/ >err.tmp 2>&1
check_err

# /etc/resolv.conf
echo -n "Copying /etc/resolv.conf ... " >>$LOG
cp /etc/resolv.conf ./pa.etc/ >err.tmp 2>&1
check_err

# /etc/services
echo -n "Copying /etc/services ... " >>$LOG
cp /etc/services ./pa.etc/ >err.tmp 2>&1
check_err

# /etc/syslog.conf
echo -n "Copying /etc/syslog.conf ... " >>$LOG
cp /etc/syslog.conf ./pa.etc/ >err.tmp 2>&1
check_err

# /etc/sysconfig/network
echo -n "Copying /etc/sysconfig/network ... " >>$LOG
cp /etc/sysconfig/network ./pa.etc/sysconfig/ >err.tmp 2>&1
check_err

# /etc/sysconfig/networking/*
echo -n "Copying /etc/sysconfig/networking/* ... " >>$LOG
cp -R /etc/sysconfig/networking ./pa.etc/sysconfig/ >err.tmp 2>&1
check_err


# --------------
# finish archive
# --------------

# finish up log file
echo "Finished:" >>$LOG
date >>$LOG

# tar and compress PA files
tar -czf $ARCHIVE_FILE *
popd >/dev/null
rm -rf $TEMP_DIR >/dev/null 2>&1
